<?php
/**
 * Systempay V2-Payment Module version 1.2.3 (revision 66856) for WooCommerce 2.x.
 *
 * Copyright (C) 2014-2015 Lyra Network and contributors
 * Support contact : supportvad@lyra-network.com
 * Author link : http://www.lyra-network.com/
 * Contributors : Alsacréations (Geoffrey Crofte http://alsacreations.fr/a-propos#geoffrey)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * @category  payment
 * @package   systempay
 * @author    Lyra Network <supportvad@lyra-network.com>
 * @copyright 2014-2015 Lyra Network and contributors
 * @license   http://www.gnu.org/licenses/old-licenses/gpl-2.0.html  GNU General Public License (GPL v2)
 * @version   1.2.3 (revision 66856)
*/

/**
 * Systempay Payment Gateway : common class.
 */
class WC_Gateway_Systempay extends WC_Payment_Gateway {

	public function __construct() {
		$this->id = 'systempay';
		$this->has_fields = false;
		$this->method_title = 'Systempay - ' . __('General configuration', 'systempay');

		// init Systempay common vars
		$this->systempay_init();

		// load the form fields
		$this->init_form_fields();

		// load the module settings
		$this->init_settings();

		$this->title = __('General configuration', 'systempay');
		$this->enabled = false;
		$this->testmode = ($this->settings['ctx_mode'] == 'TEST');
		$this->debug = (isset($this->settings['debug']) && $this->settings['debug'] == 'yes') ? true : false;

		// reset systempay common admin form action
		add_action('woocommerce_settings_start', array($this, 'systempay_reset_admin_options'));

		// update systempay admin form action
		add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));

		// return from payment platform action
		add_action('woocommerce_api_wc_gateway_systempay', array($this, 'systempay_notify_response'));
	}

	protected function systempay_init() {
		global $woocommerce;

		$this->logger = new WC_Logger();

		// init Systempay API
		$this->systempay_request = new SystempayRequest();

		$this->admin_section = 'wc_gateway_' . $this->id;
		$this->admin_tab = 'checkout';
		$this->admin_page = 'wc-settings';

		// backward compatibility
		if(version_compare($woocommerce->version, '2.1.0', '<')) {
			$this->admin_section = get_class($this);
			$this->admin_tab = 'payment_gateways';
			$this->admin_page = 'woocommerce_settings';
		}

		// admin settings page URL
		$this->admin_link = add_query_arg('section', $this->admin_section,
				add_query_arg('tab', $this->admin_tab,
						add_query_arg('page', $this->admin_page, admin_url('admin.php'))));

		// reset admin settings URL
		$this->reset_admin_link = $this->admin_link;
		$this->reset_admin_link = add_query_arg('noheader', 'true', add_query_arg('reset', 'true', $this->reset_admin_link));
		$this->reset_admin_link = wp_nonce_url($this->reset_admin_link, $this->admin_page);
	}

	/**
	 * Admin Panel Options.
	 */
	public function admin_options() {
		if (!$this->is_supported_currency()) {
			echo '<div class="inline error"><p><strong>' . __('Platform disabled', 'systempay') . ': ' . sprintf(__( '%s does not support your store currency.', 'systempay'), 'Systempay') . '</strong></p></div>';
		}

		if (get_transient($this->id . '_settings_reset')) {
			delete_transient($this->id . '_settings_reset');

			echo '<div class="inline updated"><p><strong>' . sprintf(__( 'Your %s module configuration is successfully reset.', 'systempay'), 'Systempay') . '</strong></p></div>';
		}
		?>

		<br />
		<h3>Systempay</h3>
		<p><?php echo sprintf(__('The module works by sending users to %s in order to select their payment mean and enter their payment information.', 'systempay'), 'Systempay'); ?></p>

		<table>
			<?php $this->generate_settings_html(); // generate the HTML For the settings form ?>
		</table>

		<a href="<?php echo $this->reset_admin_link; ?>"><?php _e('Reset configuration', 'systempay');?></a>

		<?php
	}


	public function systempay_reset_admin_options() {
		// if not reset action do nothing
		if(!isset($_GET['reset'])) {
			return;
		}

		// check if correct link
		if($this->admin_section != $_GET['section']) {
			return;
		}

		delete_option('woocommerce_' . $this->id . '_settings');

		// transcient flag to display reset message
		set_transient($this->id . '_settings_reset', 'true');

		wp_redirect($this->admin_link);
		die();
	}

	protected function get_supported_languages($all = false) {
		$langs = array();
		if($all) {
			$langs[''] = __('All', 'systempay');
		}

		foreach (SystempayApi::getSupportedLanguages() as $code => $label) {
			$langs[$code] = __($label, 'systempay');
		}

		return $langs;
	}

	protected function get_supported_card_types() {
		$cards = array_merge(
				array('' => __('All', 'systempay')),
				SystempayApi::getSupportedCardTypes()
		);

		return $cards;
	}

	/**
	 * Initialise Gateway Settings Form Fields
	 */
	public function init_form_fields() {
		global $woocommerce;

		if(function_exists('wc_get_log_file_path')) {
			$log_folder = dirname(wc_get_log_file_path('systempay')) . '/';
		} else {
			$log_folder = $woocommerce->plugin_path() . '/logs/';
		}
		$log_folder = str_replace('\\', '/', $log_folder);

		// get relative path
		$base_dir = str_replace('\\', '/', ABSPATH);
		if(strpos($log_folder, $base_dir) === 0) {
			$log_folder = str_replace($base_dir, '', $log_folder);
		} else {
			$base_dir = str_replace('\\', '/', dirname(ABSPATH));
			$log_folder = str_replace($base_dir, '..', $log_folder);
		}

		$this->form_fields = array(
				// module information
				'module_details' => array(
						'title' => __( 'MODULE DETAILS', 'systempay' ),
						'type' => 'title'
				),

				'developped_by' => array(
						'title' => __('Developed by', 'systempay'),
						'type' => 'text',
						'description' => '<b><a href="http://www.lyra-network.com/" target="_blank">Lyra Network</a></b>',
						'css' => 'display: none;'
				),
				'contact' => array(
						'title' => __('Contact us', 'systempay'),
						'type' => 'text',
						'description' => '<b><a href="mailto:supportvad@lyra-network.com">supportvad@lyra-network.com</a></b>',
						'css' => 'display: none;'
				),
				'contrib_version' => array(
						'title' => __('Module version', 'systempay'),
						'type' => 'text',
						'description' => '1.2.3',
						'css' => 'display: none;'
				),
				'platform_version' => array(
						'title' => __('Platform version', 'systempay'),
						'type' => 'text',
						'description' => 'V2',
						'css' => 'display: none;'
				),
				'doc_link' => array(
						'title' => '<a style="color: red;" href="'. WC_SYSTEMPAY_PLUGIN_URL.'installation_doc/Integration_Systempay_WooCommerce_2.x_v1.2.3.pdf" target="_blank">'.
									__('Click here to view the module configuration documentation.', 'systempay') . '</a>',
						'type' => 'label',
						'css' => 'font-weight: bold; text-transform: uppercase;'
				),

				'base_settings' => array(
						'title' => __( 'BASE SETTINGS', 'systempay' ),
						'type' => 'title'
				),
				'debug' => array(
						'title' => __( 'Logs', 'systempay' ),
						'label' => __('Enable / disable', 'systempay'),
						'type' => 'checkbox',
						'default' => 'yes',
						'description' => sprintf(__('Enable / disable module logs. The log file will be inside <code>%s</code>.', 'systempay'), $log_folder),
				),

				// payment platform access params
				'payment_platform_access' => array(
						'title' => __('PAYMENT PLATFORM ACCESS', 'systempay'),
						'type' => 'title'
				),
				'site_id' => array(
						'title' => __('Site ID', 'systempay'),
						'type' => 'text',
						'default' => '12345678',
						'description' => sprintf(__( 'The identifier provided by %s.', 'systempay'), 'Systempay'),
				),
				'key_test' => array(
						'title' => __('Certificate in test mode', 'systempay'),
						'type' => 'text',
						'default' => '1111111111111111',
						'description' => sprintf(__( 'Certificate provided by %s for test mode (available in %s Back Office).', 'systempay'), 'Systempay', 'Systempay')
				),
				'key_prod' => array(
						'title' => __('Certificate in production mode', 'systempay'),
						'type' => 'text',
						'default' => '2222222222222222',
						'description' => sprintf(__('Certificate provided by %s (available in %s Back Office after enabling production mode).', 'systempay'), 'Systempay', 'Systempay')
				),
				'ctx_mode' => array(
						'title' => __('Mode', 'systempay'),
						'type' => 'select',
						'default' => 'TEST',
						'options' => array(
								'TEST' => __('TEST', 'systempay'),
								'PRODUCTION' => __('PRODUCTION', 'systempay')
						),
						'description' => __('The context mode of this module.', 'systempay')
				),
				'platform_url' => array(
						'title' => __('Payment page URL', 'systempay'),
						'type' => 'text',
						'default' => 'https://paiement.systempay.fr/vads-payment/',
						'description' => __('Link to the payment page.', 'systempay'),
						'css' => 'width: 350px;'
				),
				'url_check' => array(
						'title' => sprintf(__('Instant Payment Notification URL to copy into your %s Back Office', 'systempay'), 'Systempay'),
						'type' => 'text',
						'description' => add_query_arg('wc-api', 'WC_Gateway_Systempay', network_home_url('/')),
						'css' => 'display: none;'
				),

				// payment page params
				'payment_page' => array(
						'title' => __( 'PAYMENT PAGE', 'systempay' ),
						'type' => 'title'
				),
				'language' => array(
						'title' => __('Default language', 'systempay'),
						'type' => 'select',
						'default' => 'fr',
						'options' => $this->get_supported_languages(),
						'description' => __('Default language on the payment page.', 'systempay')
				),
				'available_languages' => array(
						'title' => __('Available languages', 'systempay'),
						'type' => 'multiselect',
						'default' => array(''),
						'options' => $this->get_supported_languages(true),
						'description' => __('Languages available on the payment page. If you do not select any, all the supported languages will be available.', 'systempay')
				),
				'capture_delay' => array(
						'title' => __('Capture delay', 'systempay'),
						'type' => 'text',
						'default' => '',
						'description' => sprintf(__('The number of days before the bank capture (adjustable in your %s Back Office).', 'systempay'), 'Systempay')
				),
				'validation_mode' => array(
						'title' => __('Validation mode', 'systempay'),
						'type' => 'select',
						'default' => '',
						'options' => array(
								'' => __('Back Office configuration', 'systempay'),
								'0' => __('Automatic', 'systempay'),
								'1' => __('Manual', 'systempay')
						),
						'description' => sprintf(__('If manual is selected, you will have to confirm payments manually in your %s Back Office.', 'systempay'), 'Systempay')
				),
				'payment_cards' => array(
						'title' => __('Card Types', 'systempay'),
						'type' => 'multiselect',
						'default' => array(''),
						'options' => $this->get_supported_card_types(),
						'description' => __('The card type(s) that can be used for the payment. Select none to use platform configuration.', 'systempay')
				),

				// selective 3DS
				'selective_3ds' => array(
						'title' => __( 'SELECTIVE 3DS', 'systempay' ),
						'type' => 'title'
				),
				'3ds_min_amount' => array(
						'title' => __('Minimum amount to activate 3-DS', 'systempay'),
						'type' => 'text',
						'default' => '',
						'description' => __('Needs subscription to Selective 3-D Secure option.', 'systempay')
				),

				// return to store params
				'return_options' => array(
						'title' => __( 'RETURN OPTIONS', 'systempay' ),
						'type' => 'title'
				),
				'redirect_enabled' => array(
						'title' => __('Automatic redirection', 'systempay'),
						'label' => __('Enable / disable', 'systempay'),
						'type' => 'checkbox',
						'default' => 'no',
	 					'description' => __('If enabled, the buyer is automatically redirected to your site at the end of the payment.', 'systempay')
				),
				'redirect_success_timeout' => array(
						'title' => __('Redirection timeout on success', 'systempay'),
						'type' => 'text',
						'default' => '5',
						'description' => __('Time in seconds (0-300) before the buyer is automatically redirected to your website after a successful payment.', 'systempay')
				),
				'redirect_success_message' => array(
						'title' => __('Redirection message on success', 'systempay'),
						'type' => 'text',
						'default' => 'Redirection vers la boutique dans quelques instants...',
						'description' => __('Message displayed on the payment page prior to redirection after a successful payment.', 'systempay'),
						'css' => 'width: 350px;'
				),
				'redirect_error_timeout' => array(
						'title' => __('Redirection timeout on failure', 'systempay'),
						'type' => 'text',
						'default' => '5',
						'description' => __('Time in seconds (0-300) before the buyer is automatically redirected to your website after a declined payment.', 'systempay')
				),
				'redirect_error_message' => array(
						'title' => __('Redirection message on failure', 'systempay'),
						'type' => 'text',
						'default' => 'Redirection vers la boutique dans quelques instants...',
						'description' => __('Message displayed on the payment page prior to redirection after a declined payment.', 'systempay'),
						'css' => 'width: 350px;'
				),
				'return_mode' => array(
						'title' => __('Return mode', 'systempay'),
						'type' => 'select',
						'default' => 'GET',
						'options' => array(
								'GET' => 'GET',
								'POST' => 'POST'
						),
						'description' => __('Method that will be used for transmitting the payment result from the payment page to your shop.', 'systempay')
				)
		);
	}

	public function generate_label_html( $key, $data ) {
		$defaults = array(
				'title'             => '',
				'class'             => '',
				'css'               => '',
				'placeholder'       => '',
				'type'              => 'label',
				'description'       => ''
		);

		$data = wp_parse_args( $data, $defaults );

		ob_start();
		if ($data != null)
		?>
			<tr valign="top">
				<td class="forminp" colspan="2">
					<fieldset>
						<label class="<?php echo esc_attr( $data['class'] ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>"><?php echo wp_kses_post( $data['title'] ); ?></label>
						<p class="description"><?php echo wp_kses_post($data['description']); ?></p>
					</fieldset>
				</td>
			</tr>
		<?php
		return ob_get_clean();
	}

	/**
	 * Validate multiselect field.
	 *
	 * @return array
	 */
	public function validate_multiselect_field ($key) {
		$newValue = $_POST[$this->plugin_id . $this->id . '_' . $key];

		if(isset($newValue) && is_array($newValue) && in_array('', $newValue)) {
			return array('');
		} else {
			return parent::validate_multiselect_field ($key);
		}
	}

	/**
	 * Check if this gateway is available for the current currency.
	 */
	protected function is_supported_currency() {
		$currency = SystempayApi::findCurrencyByAlphaCode(get_woocommerce_currency());
		if($currency == null) {
			return false;
		}

		return true;
	}

	/**
	 * Check for Systempay notify Response.
	 **/
	public function systempay_notify_response() {
		@ob_clean();

		$raw_response = (array)stripslashes_deep($_REQUEST);

		$systempay_response = new SystempayResponse(
				$raw_response,
				$this->settings['ctx_mode'],
				$this->settings['key_test'],
				$this->settings['key_prod']
		);

		$from_server = $systempay_response->get('hash') != null;

		if($from_server) {
			$this->log('Response received from Systempay server URL: ' . print_r($raw_response, true));
		}

		if(!$systempay_response->isAuthentified()) {
			$this->log('Received invalid response from Systempay: authentication failed.' );

			if($from_server) {
				$this->log('SERVER URL PROCESS END');
				die($systempay_response->getOutputForGateway('auth_fail'));
			} else {
				$this->log('RETURN URL PROCESS END');
				wp_die(sprintf(__('%s response authentication failure.', 'systempay'), 'Systempay'));
			}
		} else {
			header('HTTP/1.1 200 OK');

			$this->systempay_manage_notify_response($systempay_response);
		}
	}

	/**
	 * Valid payment process : update order, send mail, ...
	 **/
	public function systempay_manage_notify_response($systempay_response) {
		global $woocommerce;

		// clear all response messages
		$this->clear_notices();

		$order_id = $systempay_response->get('order_id');
		$from_server = $systempay_response->get('hash') != null;

		$order = new WC_Order((int)$order_id);
		if (!isset($order->id) || $order->order_key !== $systempay_response->get('order_info')) {
			$this->log('Error: Order (' . $order_id . ') nor found or key does not match received invoice ID.');

			if ($from_server) {
				$this->log('SERVER URL PROCESS END');
				die($systempay_response->getOutputForGateway('order_not_found'));
			} else {
				$this->log('RETURN URL PROCESS END');
				wp_die(sprintf(__('Error : order with ID #%s cannot be found.', 'systempay'), $order_id));
			}
		}

		if($this->testmode) {
			$msg = __('<p><u>GOING INTO PRODUCTION</u></p>You want to know how to put your shop into production mode, please go to this URL: ', 'systempay');
			$msg .= '<a href="https://paiement.systempay.fr/html/faq/prod" target="_blank">https://paiement.systempay.fr/html/faq/prod</a>';

			$this->add_notice($msg);
		}

		// checkout payment URL to allow re-order
		$error_url = $woocommerce->cart->get_checkout_url();

		// backward compatibility
		if(version_compare($woocommerce->version, '2.1.0', '<')) {
			$error_url = $order->get_cancel_order_url();
		}

		if($this->is_new_order($order, $systempay_response->get('trans_id'))) {
			// order not processed yet or a failed payment (re-order)

			// store transaction details
			update_post_meta((int) $order_id, 'Transaction ID', $systempay_response->get('trans_id'));
			update_post_meta((int) $order_id, 'Card number', $systempay_response->get('card_number'));
			update_post_meta((int) $order_id, 'Payment mean', $systempay_response->get('card_brand'));

			$expiry = str_pad($systempay_response->get('expiry_month'), 2, '0', STR_PAD_LEFT) . '/' . $systempay_response->get('expiry_year');
			if(! $systempay_response->get('expiry_month')) {
				$expiry = '';
			}
			update_post_meta((int) $order_id, 'Card expiry', $expiry);

			if($systempay_response->isAcceptedPayment()) {
				$this->log('Payment successfull, let\'s save order #' . $order_id);

				// payment completed
				$note = __('Payment completed successfully.', 'systempay');
				$note .= ' ';
				$note .= sprintf(__('Transaction %s.', 'systempay'), $systempay_response->get('trans_id'));
				$order->add_order_note($note);
				$order->payment_complete();

				if ($from_server) {
					$this->log('Payment completed successfully by server URL call.');
					$this->log('SERVER URL PROCESS END');

					die ($systempay_response->getOutputForGateway('payment_ok'));
				} else {
					$this->log('Warning ! IPN URL call does not worked. Payment completed by return URL call.');

					if($this->testmode) {
						$ipn_url_warn = sprintf(__('The automatic notification (peer to peer connection between the payment platform and your shopping cart solution) hasn\'t worked. Have you correctly set up the notification URL in the %s Back Office ?', 'systempay'), 'Systempay');
						$ipn_url_warn .= '<br />';
						$ipn_url_warn .= __('For understanding the problem, please read the documentation of the module : <br />&nbsp;&nbsp;&nbsp;- Chapter &laquo;To read carefully before going further&raquo;<br />&nbsp;&nbsp;&nbsp;- Chapter &laquo;Notification URL settings&raquo;', 'systempay');

						$this->add_notice($ipn_url_warn, 'error');
					}

					$this->log('RETURN URL PROCESS END');
					wp_redirect($this->get_return_url($order));
					die();
				}
			} else {
				$note = $systempay_response->getLogString();
				if(!$systempay_response->isCancelledPayment()) {
					$note .= ' ';
					$note .= sprintf(__('Transaction %s.', 'systempay'), $systempay_response->get('trans_id'));
				}
				$order->add_order_note($note);
				$order->update_status('failed');

				$this->log('Payment failed or cancelled. ' . $systempay_response->getLogString());

				if ($from_server) {
					$this->log('SERVER URL PROCESS END');
					die($systempay_response->getOutputForGateway('payment_ko'));
				} else {
					if (!$systempay_response->isCancelledPayment()) {
						$this->add_notice(__('Your payment was not accepted. Please, try to re-order.', 'systempay'), 'error');
					}

					$this->log('RETURN URL PROCESS END');
					wp_redirect($error_url);
					die();
				}
			}
		} else {
			$this->log('Order #' . $order_id . ' is already processed. Just show payment result.' );

			if($systempay_response->isAcceptedPayment() && ($order->status === 'completed' || $order->status === 'processing')) {
				$this->log('Payment successfull reconfirmed.');

				// order success registered and payment succes received
				if ($from_server) {
					$this->log('SERVER URL PROCESS END');
					die ($systempay_response->getOutputForGateway('payment_ok_already_done'));
				} else {
					$this->log('RETURN URL PROCESS END');
					wp_redirect($this->get_return_url($order));
					die();
				}
			} elseif(!$systempay_response->isAcceptedPayment() && ($order->status === 'failed' || $order->status === 'cancelled')) {
				$this->log('Payment failed reconfirmed.');

				// order failure registered and payment error received
				if ($from_server) {
					$this->log('SERVER URL PROCESS END');
					die($systempay_response->getOutputForGateway('payment_ko_already_done'));
				} else {
					$this->log('RETURN URL PROCESS END');

					if (!$systempay_response->isCancelledPayment()) {
						$this->add_notice(__('Your payment was not accepted. Please, try to re-order.', 'systempay'), 'error');
					}

					wp_redirect($error_url);
					die();
				}
			} else {
				$this->log('Error ! Invalid payment result received for already saved order. Payment result : ' . $systempay_response->get('result') . ', Order status : ' . $order->status);

				// registered order status not match payment result
				if ($from_server) {
					$this->log('SERVER URL PROCESS END');
					die($systempay_response->getOutputForGateway('payment_ko_on_order_ok'));
				} else {
					$this->log('RETURN URL PROCESS END');
					wp_die(sprintf(__('Error : invalid payment code received for already processed order (%s).', 'systempay'), $order_id));
				}
			}
		}
	}

	private function is_new_order($order, $trs_id) {
		if ($order->status === 'pending') {
			return true;
		}

		if ($order->status === 'failed' || $order->status === 'cancelled') {
			return get_post_meta((int) $order->id, 'Transaction ID', true) !== $trs_id;
		}

		return false;
	}

	protected function log($msg) {
		if (!$this->debug) {
			return;
		}

		$this->logger->add('systempay', $msg);
	}

	protected function clear_notices() {
		global $woocommerce;

		if(function_exists('wc_clear_notices')) {
			wc_clear_notices();
		} else {
			$woocommerce->clear_messages();
		}
	}

	protected function add_notice($msg, $type='success') {
		global $woocommerce;

		if(function_exists('wc_add_notice')) {
			wc_add_notice($msg, $type);
		} else {
			if($type == 'error') {
				$woocommerce->add_error($msg);
			} else {
				$woocommerce->add_message($msg);
			}
		}
	}

}